//  
//  Copyright (C) 2011 Tyler Burton
//	www.tylerburton.ca
// 
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
//

package ca.tylerburton.mymediaplayer;

import java.awt.BorderLayout;
import java.awt.EventQueue;
import java.awt.GridLayout;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.util.concurrent.TimeUnit;

import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.UIManager;

import org.gstreamer.Bus;
import org.gstreamer.ElementFactory;
import org.gstreamer.Gst;
import org.gstreamer.GstObject;
import org.gstreamer.Pipeline;
import org.gstreamer.State;
import org.gstreamer.TagList;
import org.gstreamer.elements.PlayBin;

/**
 * A graphical user interface that allows a user to play back audio files.
 * Powered by GStreamer.
 * 
 * @author Tyler Burton (software@tylerburton.ca)
 * @version March 13, 2011
 *
 */
public class MyMediaPlayer {
	
	/*
	 * Create globals
	 */
	
	private PlayBin _playbin = null; // Default audio PlayBin
	private JProgressBar _progBar = null; // Progress bar on GUI
	JLabel _lblArtist = null; // Label to hold artist name
	JLabel _lblAlbum = null; // Label to hold album name
	JLabel _lblTrack = null; // Label to hold track name
	JLabel _lblDuration = null; // Label to hold the song position
	TimeUnit _scaleUnit = TimeUnit.SECONDS; // The scale unit to use in time calculations
	
	/**
	 * Creates a Swing GUI for media playback
	 */
	public MyMediaPlayer() {
		EventQueue.invokeLater(new Runnable() {
			public void run() {
				
				// Try and mimic the system look and feel
				try {
					UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
				} catch (Exception e) {}
				
				/*
				 * Create the Swing GUI
				 */
				
				final JFrame f = new JFrame("My Media Player");
				f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
				
				f.setLayout(new BorderLayout());
				
				JMenuBar menuBar = new JMenuBar();
				
				JMenu menu = new JMenu("File"); 
				menuBar.add(menu);
				
				JMenuItem menuItem = new JMenuItem("Open");
				menu.add(menuItem);
				
				menuItem.addActionListener(new ActionListener() {
					public void actionPerformed(ActionEvent arg0) {
						final JFileChooser fc = new JFileChooser();
						
						if(fc.showOpenDialog(f) == JFileChooser.APPROVE_OPTION) {
							playFile(fc.getSelectedFile());
						}
					}
				});
				
				f.setJMenuBar(menuBar);
				
				_progBar = new JProgressBar();
				f.add(_progBar, BorderLayout.NORTH);
				
				_progBar.setMaximum(100);
				_progBar.setMinimum(0);
				
				/*
				 * Allow the user to click a location on the progress bar to seek to that spot
				 */
				_progBar.addMouseListener(new MouseListener() {
					public void mouseClicked(MouseEvent arg0) {
						// Avoid divide by zero exception
						if(_progBar.getWidth() == 0)
							return;
						
						Point mouse = arg0.getPoint();
						double perc = (double)mouse.x/(double)_progBar.getWidth();
						long duration = _playbin.queryDuration(_scaleUnit);
			
						// Seek to that percentage of the song
						_playbin.seek((long) (duration*perc), _scaleUnit);
					}

					public void mouseEntered(MouseEvent arg0) {
					}

					public void mouseExited(MouseEvent arg0) {
					}

					public void mousePressed(MouseEvent arg0) {
					}

					public void mouseReleased(MouseEvent arg0) {
					}
				});
							
				JPanel panel = new JPanel(new GridLayout(0,2));
				f.add(panel);
				
				panel.add(new JLabel("Time: "));
				_lblDuration = new JLabel("");
				panel.add(_lblDuration);
				
				panel.add(new JLabel("Artist:"));
				_lblArtist = new JLabel("                  "); // Padding
				panel.add(_lblArtist);
				
				panel.add(new JLabel("Album: "));
				_lblAlbum = new JLabel("");
				panel.add(_lblAlbum);
				
				panel.add(new JLabel("Track: "));
				_lblTrack = new JLabel("");
				panel.add(_lblTrack);
				
				/*
				 * Pack the GUI. If the width ends up being less than 
				 * 400px then widen to 400px
				 */
				
				f.pack();
				
				if(f.getWidth() < 400)
					f.setSize(400, f.getHeight());
				f.setVisible(true);
			}
		});
		
		// Initialize GStreamer
		initGstreamer();
		
		// Add end of stream (EOS) action listener
		_playbin.getBus().connect(new Bus.EOS() {
			public void endOfStream(GstObject arg0) {
				_lblArtist.setText("SONG");
				_lblAlbum.setText("HAS");
				_lblTrack.setText("FINISHED");
			}
		});
		
		// Add meta data action listener
		_playbin.getBus().connect(new Bus.TAG() {
			public void tagsFound(GstObject source, TagList tagList) {
				for(String tagName : tagList.getTagNames()) {
					if(tagName.equalsIgnoreCase("artist")) {
						_lblArtist.setText((String) tagList.getValues(tagName).get(0));
					}else if(tagName.equalsIgnoreCase("album")) {
						_lblAlbum.setText((String) tagList.getValues(tagName).get(0));
					}else if(tagName.equalsIgnoreCase("title")) {
						_lblTrack.setText((String) tagList.getValues(tagName).get(0));
					}
				}
			}
		});
		
		// Start main loop
		Gst.main();
		_playbin.setState(State.NULL);
	}
	
	/**
	 * Initialize GStreamer and create default PlayBin
	 */
	private void initGstreamer() {
		_playbin = new PlayBin("AudioPlayer");
		_playbin.setVideoSink(ElementFactory.make("fakesink", "videosink"));
		startPoll();
	}
	
	/**
	 * Set the file as the target to play
	 * @param file The file to play
	 */
	private synchronized void playFile(File file) {
		_playbin.stop();
		
		if(file.exists()) {
			_lblArtist.setText("");
			_lblAlbum.setText("");
			_lblTrack.setText("");
			_playbin.setInputFile(file);
			_playbin.setState(State.PLAYING);
		}else{
			_lblArtist.setText("FILE NOT FOUND");
			_lblAlbum.setText("FILE NOT FOUND");
			_lblTrack.setText("FILE NOT FOUND");
		}
	}
	
	/**
	 * Poll the GStreamer state and update the duration time label and progress bar
	 */
	private void startPoll() {
		Thread task = new Thread() {
			public void run() {
				while(true) {
					try{
						Pipeline pipeline = _playbin;
						
						long position = pipeline.queryPosition(_scaleUnit);
						long duration = pipeline.queryDuration(_scaleUnit);
						
						_progBar.setMaximum(100);
						_progBar.setValue((int) ((position*100)/duration));
						
						int intCurMin = (int) (position/60);
						String curMin = null;
						
						if(intCurMin < 10)
							curMin = "0" + intCurMin;
						else
							curMin = "" + intCurMin;
						
						int intCurSec = (int) (position - (60*intCurMin));
						
						String curSec = null;
						if(intCurSec < 10)
							curSec = "0" + intCurSec;
						else
							curSec = "" + intCurSec;
						
						int intMaxMin = (int) (duration/60);
						
						String maxMin = null;
						if(intMaxMin < 10)
							maxMin = "0" + intMaxMin;
						else
							maxMin = "" + intMaxMin;
						
						
						int intMaxSec = (int) (duration - (60*intMaxMin));
						
						String maxSec = null;
						if(intMaxSec < 10)
							maxSec = "0" + intMaxSec;
						else
							maxSec = "" + intMaxSec;
						
						_lblDuration.setText(curMin + ":" + curSec + "/" + maxMin + ":" + maxSec);
						
					}catch(Exception e) {}
					
					try {
						Thread.sleep(1000);
					}catch(Exception e) {}
				}
			}
		};
		
		task.start();
	}
	
}
